<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Order;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Store;
use App\Models\StockItem;
use App\Models\User;
use App\Services\OrderService;
use App\Exceptions\InsufficientStockException;
use Illuminate\Foundation\Testing\RefreshDatabase;

class OrderStockTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Store $store;
    protected Product $product;
    protected ProductVariant $variant;
    protected StockItem $stockItem;
    protected OrderService $orderService;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
        $this->store = Store::factory()->create(['is_active' => true]);
        $this->product = Product::factory()->create(['is_available' => true]);
        $this->variant = ProductVariant::factory()->create([
            'product_id' => $this->product->id,
            'price' => 100,
            'is_available' => true,
        ]);

        $this->stockItem = StockItem::create([
            'product_id' => $this->product->id,
            'variant_id' => $this->variant->id,
            'store_id' => $this->store->id,
            'quantity' => 100,
            'reserved_quantity' => 0,
        ]);

        $this->orderService = app(OrderService::class);
    }

    /** @test */
    public function it_reserves_stock_when_creating_order()
    {
        $orderData = $this->makeOrderData(10);

        $order = $this->orderService->createOrder($orderData);

        $this->stockItem->refresh();

        // Stock physique inchangé
        $this->assertEquals(100, $this->stockItem->quantity);

        // Stock réservé incrémenté
        $this->assertEquals(10, $this->stockItem->reserved_quantity);

        // Stock disponible calculé
        $this->assertEquals(90, $this->stockItem->available_quantity);

        // Commande créée avec bon statut
        $this->assertEquals('EN_ATTENTE_TRAITEMENT', $order->status);
    }

    /** @test */
    public function it_refuses_order_when_insufficient_available_stock()
    {
        // Réserver 90 unités (stock disponible = 10)
        $this->stockItem->reserved_quantity = 90;
        $this->stockItem->save();

        $orderData = $this->makeOrderData(20); // Demande 20 > 10 disponibles

        $this->expectException(InsufficientStockException::class);

        $this->orderService->createOrder($orderData);

        // Vérifier qu'aucune commande n'a été créée
        $this->assertEquals(0, Order::count());
    }

    /** @test */
    public function it_releases_reserved_stock_when_cancelling_order()
    {
        $orderData = $this->makeOrderData(10);
        $order = $this->orderService->createOrder($orderData);

        $this->stockItem->refresh();
        $this->assertEquals(10, $this->stockItem->reserved_quantity);

        // Annuler la commande
        $this->orderService->cancelOrder($order->id, 'Test annulation', $this->user->id);

        $this->stockItem->refresh();

        // Stock physique toujours inchangé
        $this->assertEquals(100, $this->stockItem->quantity);

        // Stock réservé libéré
        $this->assertEquals(0, $this->stockItem->reserved_quantity);

        // Stock disponible restauré
        $this->assertEquals(100, $this->stockItem->available_quantity);
    }

    /** @test */
    public function it_decrements_physical_stock_when_delivering_order()
    {
        $orderData = $this->makeOrderData(10);
        $order = $this->orderService->createOrder($orderData);

        $this->stockItem->refresh();
        $this->assertEquals(100, $this->stockItem->quantity);
        $this->assertEquals(10, $this->stockItem->reserved_quantity);

        // Livrer la commande
        $this->orderService->deliverOrder($order->id, 'Test livraison', $this->user->id);

        $this->stockItem->refresh();

        // Stock physique décrémenté
        $this->assertEquals(90, $this->stockItem->quantity);

        // Stock réservé libéré
        $this->assertEquals(0, $this->stockItem->reserved_quantity);

        // Stock disponible correct
        $this->assertEquals(90, $this->stockItem->available_quantity);

        // Statut livré
        $order->refresh();
        $this->assertEquals('LIVREE', $order->status);
        $this->assertNotNull($order->delivered_at);
    }

    /** @test */
    public function it_prevents_race_condition_on_concurrent_orders()
    {
        // Stock initial: 100, réservé: 0, disponible: 100

        // Commande 1: 60 unités
        $order1 = $this->orderService->createOrder($this->makeOrderData(60));
        $this->stockItem->refresh();
        $this->assertEquals(60, $this->stockItem->reserved_quantity);

        // Commande 2: 50 unités (devrait échouer car disponible = 40)
        $this->expectException(InsufficientStockException::class);
        $this->orderService->createOrder($this->makeOrderData(50));
    }

    /** @test */
    public function it_handles_multiple_items_in_order()
    {
        // Créer un deuxième produit
        $variant2 = ProductVariant::factory()->create([
            'product_id' => $this->product->id,
            'price' => 200,
            'is_available' => true,
        ]);

        $stockItem2 = StockItem::create([
            'product_id' => $this->product->id,
            'variant_id' => $variant2->id,
            'store_id' => $this->store->id,
            'quantity' => 50,
            'reserved_quantity' => 0,
        ]);

        $orderData = [
            'user_id' => $this->user->id,
            'customer_first_name' => 'Test',
            'customer_last_name' => 'User',
            'customer_phone' => '0123456789',
            'delivery_street' => '123 Test St',
            'store_id' => $this->store->id,
            'items' => [
                ['variant_id' => $this->variant->id, 'quantity' => 10],
                ['variant_id' => $variant2->id, 'quantity' => 5],
            ],
        ];

        $order = $this->orderService->createOrder($orderData);

        $this->stockItem->refresh();
        $stockItem2->refresh();

        // Vérifier les deux réservations
        $this->assertEquals(10, $this->stockItem->reserved_quantity);
        $this->assertEquals(5, $stockItem2->reserved_quantity);
    }

    /** @test */
    public function it_refuses_entire_order_if_one_item_insufficient()
    {
        // Stock variant1: 100 disponibles
        // Stock variant2: 5 disponibles (insuffisant pour 10)

        $variant2 = ProductVariant::factory()->create([
            'product_id' => $this->product->id,
            'price' => 200,
            'is_available' => true,
        ]);

        $stockItem2 = StockItem::create([
            'product_id' => $this->product->id,
            'variant_id' => $variant2->id,
            'store_id' => $this->store->id,
            'quantity' => 5,
            'reserved_quantity' => 0,
        ]);

        $orderData = [
            'user_id' => $this->user->id,
            'customer_first_name' => 'Test',
            'customer_last_name' => 'User',
            'customer_phone' => '0123456789',
            'delivery_street' => '123 Test St',
            'store_id' => $this->store->id,
            'items' => [
                ['variant_id' => $this->variant->id, 'quantity' => 10],
                ['variant_id' => $variant2->id, 'quantity' => 10], // Insuffisant
            ],
        ];

        $this->expectException(InsufficientStockException::class);
        $this->orderService->createOrder($orderData);

        // Vérifier qu'AUCUNE réservation n'a été faite
        $this->stockItem->refresh();
        $stockItem2->refresh();

        $this->assertEquals(0, $this->stockItem->reserved_quantity);
        $this->assertEquals(0, $stockItem2->reserved_quantity);

        // Vérifier qu'aucune commande n'a été créée
        $this->assertEquals(0, Order::count());
    }

    protected function makeOrderData(int $quantity): array
    {
        return [
            'user_id' => $this->user->id,
            'customer_first_name' => 'Test',
            'customer_last_name' => 'User',
            'customer_phone' => '0123456789',
            'customer_email' => 'test@example.com',
            'delivery_street' => '123 Test St',
            'delivery_city' => 'Paris',
            'delivery_postal_code' => '75001',
            'store_id' => $this->store->id,
            'items' => [
                [
                    'variant_id' => $this->variant->id,
                    'quantity' => $quantity,
                ],
            ],
        ];
    }
}
