@extends('layouts.admin')

@section('title', 'Creer un produit')
@section('page-title', 'Produits')

@section('content')
<div class="bg-white rounded-lg shadow-sm p-6">
    <div class="flex items-center justify-between mb-6">
        <h2 class="text-lg font-semibold">Creer un produit</h2>
        <a href="{{ route('admin.products.index') }}" class="text-sm text-gray-600 hover:text-primary">Retour</a>
    </div>

    @if($errors->any())
        <div class="mb-4 rounded-lg border border-red-200 bg-red-50 p-4 text-sm text-red-700">
            <ul class="list-disc pl-4 space-y-1">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    @php
        $categoriesForJs = $categories->map(function ($category) {
            return [
                'id' => $category->id,
                'name' => $category->name,
                'attributes' => $category->attributes->map(function ($attribute) {
                    return [
                        'id' => $attribute->id,
                        'name' => $attribute->name,
                        'type' => $attribute->type,
                        'required' => (bool) $attribute->required,
                        'options' => $attribute->options ?? [],
                    ];
                })->values(),
            ];
        })->values();
    @endphp

    <form method="POST" action="{{ route('admin.products.store') }}" enctype="multipart/form-data" class="space-y-6">
        @csrf

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Nom *</label>
                <input type="text" name="name" value="{{ old('name') }}" required
                       class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent">
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Categorie *</label>
                <select name="category_id" id="category_id" required
                        class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent">
                    <option value="">Selectionner</option>
                    @foreach($categories as $category)
                        <option value="{{ $category->id }}" {{ old('category_id') == $category->id ? 'selected' : '' }}>
                            {{ $category->name }}
                        </option>
                    @endforeach
                </select>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Prix *</label>
                <input type="number" name="base_price" step="0.01" min="0" value="{{ old('base_price') }}" required
                       class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent">
            </div>

            <div class="flex items-center gap-6">
                <label class="inline-flex items-center gap-3">
                    <input type="checkbox" name="is_active" value="1" id="is_active"
                           {{ old('is_active', true) ? 'checked' : '' }}
                           class="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary">
                    <span class="text-sm text-gray-700">Produit actif</span>
                </label>
                <label class="inline-flex items-center gap-3">
                    <input type="checkbox" name="is_featured" value="1" id="is_featured"
                           {{ old('is_featured') ? 'checked' : '' }}
                           class="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary">
                    <span class="text-sm text-gray-700">Produit favori (affiche sur la page d'accueil)</span>
                </label>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Image principale</label>
                <input type="file" name="thumbnail" accept="image/*"
                       class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm">
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Images secondaires</label>
                <input type="file" name="images[]" accept="image/*" multiple
                       class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm">
            </div>

            <div class="lg:col-span-2">
                <label class="block text-sm font-medium text-gray-700 mb-2">Description courte</label>
                <textarea name="short_desc" rows="2"
                          class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent">{{ old('short_desc') }}</textarea>
            </div>

            <div class="lg:col-span-2">
                <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                <textarea name="description" rows="5"
                          class="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent">{{ old('description') }}</textarea>
            </div>
        </div>

        <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">Attributs</label>
            <p class="text-xs text-gray-500 mb-3">Choisissez une categorie pour afficher les attributs lies.</p>
            <div id="attribute-fields" class="grid grid-cols-1 md:grid-cols-2 gap-4"></div>
        </div>

        <div class="flex items-center justify-end gap-3">
            <a href="{{ route('admin.products.index') }}"
               class="px-4 py-2 text-sm text-gray-600 hover:text-primary">Annuler</a>
            <button type="submit"
                    class="px-4 py-2 text-sm font-medium text-white bg-primary rounded-lg hover:bg-primary/90">Creer</button>
        </div>
    </form>
</div>

<script>
    const categories = @json($categoriesForJs);
    const attributeValues = @json(old('attribute_values', $attributeValues ?? []));
    const categorySelect = document.getElementById('category_id');
    const attributeContainer = document.getElementById('attribute-fields');

    const renderAttributes = () => {
        const selectedId = parseInt(categorySelect.value, 10);
        const selectedCategory = categories.find(category => category.id === selectedId);

        attributeContainer.innerHTML = '';

        if (!selectedCategory || selectedCategory.attributes.length === 0) {
            const empty = document.createElement('p');
            empty.className = 'text-xs text-gray-500';
            empty.textContent = 'Aucun attribut pour cette categorie.';
            attributeContainer.appendChild(empty);
            return;
        }

        selectedCategory.attributes.forEach(attribute => {
            const wrapper = document.createElement('div');

            const label = document.createElement('label');
            label.className = 'block text-sm font-medium text-gray-700 mb-2';
            label.textContent = attribute.name + (attribute.required ? ' *' : '');
            wrapper.appendChild(label);

            let field;
            if (attribute.type === 'select') {
                field = document.createElement('select');
                field.className = 'w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent';
                const placeholder = document.createElement('option');
                placeholder.value = '';
                placeholder.textContent = 'Selectionner';
                field.appendChild(placeholder);
                (attribute.options || []).forEach(option => {
                    const opt = document.createElement('option');
                    opt.value = option;
                    opt.textContent = option;
                    field.appendChild(opt);
                });
            } else {
                field = document.createElement('input');
                field.type = attribute.type === 'number' ? 'number' : 'text';
                field.step = attribute.type === 'number' ? '0.01' : '';
                field.className = 'w-full rounded-lg border border-gray-300 px-3 py-2 text-sm focus:ring-2 focus:ring-primary focus:border-transparent';
            }

            field.name = `attribute_values[${attribute.id}]`;
            field.required = !!attribute.required;
            if (attributeValues && attributeValues[attribute.id] !== undefined) {
                field.value = attributeValues[attribute.id];
            }

            wrapper.appendChild(field);
            attributeContainer.appendChild(wrapper);
        });
    };

    categorySelect.addEventListener('change', renderAttributes);
    renderAttributes();
</script>
@endsection
