<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'order_number' => $this->order_number,

            // Informations client
            'customer' => [
                'first_name' => $this->customer_first_name,
                'last_name' => $this->customer_last_name,
                'full_name' => $this->customer_full_name,
                'phone' => $this->customer_phone,
                'email' => $this->customer_email,
            ],

            // Adresse de livraison
            'delivery_address' => [
                'street' => $this->delivery_street,
                'city' => $this->delivery_city,
                'postal_code' => $this->delivery_postal_code,
                'country' => $this->delivery_country,
                'full_address' => $this->delivery_address,
            ],

            // Magasin assigné
            'store' => $this->when($this->relationLoaded('store'), function () {
                return [
                    'id' => $this->store->id,
                    'name' => $this->store->name,
                    'city' => $this->store->city,
                    'phone' => $this->store->phone,
                ];
            }),

            // Montants
            'pricing' => [
                'subtotal' => (float) $this->subtotal,
                'tax_amount' => (float) $this->tax_amount,
                'shipping_cost' => (float) $this->shipping_cost,
                'total_amount' => (float) $this->total_amount,
            ],

            // Statut
            'status' => $this->status,
            'status_label' => $this->getStatusLabel(),
            'payment_method' => $this->payment_method,
            'payment_method_label' => 'Paiement à la livraison',
            'is_paid' => (bool) $this->is_paid,

            // Items
            'items' => OrderItemResource::collection($this->whenLoaded('items')),
            'items_count' => $this->when(
                $this->relationLoaded('items'),
                $this->items->count()
            ),

            // Notes
            'customer_notes' => $this->customer_notes,
            'admin_notes' => $this->when(
                $request->user()?->isAdmin(),
                $this->admin_notes
            ),

            // Historique statuts (admin seulement)
            'status_history' => $this->when(
                $request->user()?->isAdmin() && $this->relationLoaded('statusHistory'),
                OrderStatusHistoryResource::collection($this->statusHistory)
            ),

            // Dates
            'created_at' => $this->created_at?->toIso8601String(),
            'updated_at' => $this->updated_at?->toIso8601String(),
            'delivered_at' => $this->delivered_at?->toIso8601String(),
        ];
    }

    /**
     * Get status label in French
     */
    protected function getStatusLabel(): string
    {
        return match($this->status) {
            'PENDING' => 'En attente',
            'CONFIRMED' => 'Confirmée',
            'PREPARING' => 'En préparation',
            'READY' => 'Prête pour livraison',
            'DELIVERED' => 'Livrée',
            'CANCELLED' => 'Annulée',
            default => $this->status,
        };
    }
}
