<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class CreateProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        // Vérifier que l'utilisateur est admin (via middleware)
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $rules = [
            // Informations générales
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:products,slug',
            'description' => 'required|string|max:5000',
            'short_desc' => 'nullable|string|max:500',
            'base_price' => 'required|numeric|min:0',

            // Caractéristiques matelas
            'type' => 'nullable|in:FOAM,SPRING,MEMORY_FOAM,LATEX,HYBRID',
            'firmness' => 'nullable|in:SOFT,MEDIUM,FIRM,EXTRA_FIRM',
            'height' => 'nullable|integer|min:1|max:100',
            'material' => 'nullable|string|max:500',
            'warranty' => 'nullable|integer|min:0|max:30',

            // SEO
            'meta_title' => 'nullable|string|max:255',
            'meta_desc' => 'nullable|string|max:500',

            // Médias
            'images' => 'nullable|array',
            'images.*' => 'nullable|url',
            'thumbnail' => 'nullable|url',

            // Statut
            'is_active' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',

            // Catégorie
            'category_id' => 'nullable|exists:categories,id',

            // Variantes (tailles)
            'variants' => 'required|array|min:1',
            'variants.*.size' => 'required|string|max:50',
            'variants.*.price' => 'required|numeric|min:0',
            'variants.*.sku' => 'nullable|string|max:100|unique:product_variants,sku',
        ];

        // Si c'est une mise à jour (PUT/PATCH), ignorer le slug du produit actuel
        if ($this->isMethod('PUT') || $this->isMethod('PATCH')) {
            $productId = $this->route('id') ?? $this->route('product');
            $rules['slug'] = 'nullable|string|max:255|unique:products,slug,' . $productId;
        }

        return $rules;
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'name.required' => 'Le nom du produit est obligatoire',
            'description.required' => 'La description est obligatoire',
            'base_price.required' => 'Le prix de base est obligatoire',
            'base_price.min' => 'Le prix doit être positif',

            'type.in' => 'Le type de matelas est invalide',
            'firmness.in' => 'La fermeté est invalide',
            'height.min' => 'La hauteur doit être au minimum 1 cm',
            'height.max' => 'La hauteur ne peut pas dépasser 100 cm',
            'warranty.max' => 'La garantie ne peut pas dépasser 30 ans',

            'category_id.exists' => 'La catégorie sélectionnée n\'existe pas',

            'variants.required' => 'Au moins une variante (taille) est requise',
            'variants.min' => 'Au moins une variante (taille) est requise',
            'variants.*.size.required' => 'La taille est obligatoire',
            'variants.*.price.required' => 'Le prix de la variante est obligatoire',
            'variants.*.sku.unique' => 'Ce SKU existe déjà',
        ];
    }
}
