<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class CreateOrderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        // Pas besoin d'authentification pour créer une commande
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        return [
            // Informations client (obligatoires)
            'customer_first_name' => 'required|string|max:255',
            'customer_last_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:20|regex:/^[0-9+\s\-()]+$/',
            'customer_email' => 'nullable|email|max:255',

            // Adresse de livraison (obligatoire)
            'delivery_street' => 'required|string|max:500',
            'delivery_city' => 'required|string|max:255',
            'delivery_postal_code' => 'required|string|max:20',
            'delivery_country' => 'nullable|string|max:100',

            // Magasin de livraison (obligatoire)
            'store_id' => 'required|exists:stores,id',

            // Items de commande (minimum 1)
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.variant_id' => 'required|exists:product_variants,id',
            'items.*.quantity' => 'required|integer|min:1|max:100',

            // Notes client (optionnel)
            'customer_notes' => 'nullable|string|max:1000',

            // ID utilisateur si connecté (optionnel)
            'user_id' => 'nullable|exists:users,id',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'customer_first_name.required' => 'Le prénom est obligatoire',
            'customer_last_name.required' => 'Le nom est obligatoire',
            'customer_phone.required' => 'Le numéro de téléphone est obligatoire',
            'customer_phone.regex' => 'Le format du numéro de téléphone est invalide',
            'customer_email.email' => 'L\'adresse email est invalide',

            'delivery_street.required' => 'L\'adresse de livraison est obligatoire',
            'delivery_city.required' => 'La ville est obligatoire',
            'delivery_postal_code.required' => 'Le code postal est obligatoire',

            'store_id.required' => 'Le magasin de livraison est obligatoire',
            'store_id.exists' => 'Le magasin sélectionné n\'existe pas',

            'items.required' => 'La commande doit contenir au moins un article',
            'items.min' => 'La commande doit contenir au moins un article',
            'items.*.product_id.exists' => 'Le produit sélectionné n\'existe pas',
            'items.*.variant_id.exists' => 'La variante sélectionnée n\'existe pas',
            'items.*.quantity.min' => 'La quantité minimale est 1',
            'items.*.quantity.max' => 'La quantité maximale est 100',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes(): array
    {
        return [
            'customer_first_name' => 'prénom',
            'customer_last_name' => 'nom',
            'customer_phone' => 'téléphone',
            'customer_email' => 'email',
            'delivery_street' => 'adresse',
            'delivery_city' => 'ville',
            'delivery_postal_code' => 'code postal',
            'store_id' => 'magasin',
            'items' => 'articles',
        ];
    }
}
