<?php

namespace App\Http\Controllers;

use App\Services\CartService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class CartController extends Controller
{
    public function __construct(
        private CartService $cartService
    ) {}

    /**
     * Display cart page
     */
    public function index(): View
    {
        $cart = $this->cartService->getCartWithProducts();

        return view('cart.index', [
            'cart' => $cart,
        ]);
    }

    /**
     * Add item to cart (AJAX)
     */
    public function add(Request $request): JsonResponse
    {
        $request->validate([
            'variant_id' => 'required|integer|exists:product_variants,id',
            'quantity' => 'required|integer|min:1',
        ]);

        try {
            $cart = $this->cartService->addItem(
                $request->variant_id,
                $request->quantity
            );

            return response()->json([
                'success' => true,
                'message' => 'Produit ajouté au panier',
                'cart' => [
                    'total_items' => $cart['total_items'],
                    'total_price' => $cart['total_price'],
                ],
            ]);
        } catch (\InvalidArgumentException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Update item quantity (AJAX)
     */
    public function update(Request $request): JsonResponse
    {
        $request->validate([
            'variant_id' => 'required|integer|exists:product_variants,id',
            'quantity' => 'required|integer|min:0',
        ]);

        try {
            $cart = $this->cartService->updateQuantity(
                $request->variant_id,
                $request->quantity
            );

            return response()->json([
                'success' => true,
                'message' => 'Panier mis à jour',
                'cart' => [
                    'total_items' => $cart['total_items'],
                    'total_price' => $cart['total_price'],
                ],
            ]);
        } catch (\InvalidArgumentException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Remove item from cart (AJAX)
     */
    public function remove(Request $request): JsonResponse
    {
        $request->validate([
            'variant_id' => 'required|integer|exists:product_variants,id',
        ]);

        try {
            $cart = $this->cartService->removeItem($request->variant_id);

            return response()->json([
                'success' => true,
                'message' => 'Produit retiré du panier',
                'cart' => [
                    'total_items' => $cart['total_items'],
                    'total_price' => $cart['total_price'],
                ],
            ]);
        } catch (\InvalidArgumentException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Clear cart (AJAX)
     */
    public function clear(): JsonResponse
    {
        $this->cartService->clearCart();

        return response()->json([
            'success' => true,
            'message' => 'Panier vidé',
            'cart' => [
                'total_items' => 0,
                'total_price' => 0,
            ],
        ]);
    }

    /**
     * Get cart data (AJAX)
     */
    public function get(): JsonResponse
    {
        $cart = $this->cartService->getCart();

        return response()->json([
            'success' => true,
            'cart' => $cart,
        ]);
    }
}
