<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Store;
use Illuminate\Http\Request;

class StoreController extends Controller
{
    public function index()
    {
        // Par défaut, filtrer les magasins actifs pour l'API publique
        // Sauf si include_inactive=true (réservé admin)
        $query = Store::query();

        if (!request()->has('include_inactive') || !request()->boolean('include_inactive')) {
            $query->where('is_active', true);
        }

        $stores = $query->get();

        return response()->json([
            'data' => $stores,
        ]);
    }

    public function show($id)
    {
        $store = Store::findOrFail($id);

        return response()->json([
            'data' => $store,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'street' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'postal_code' => 'required|string|max:20',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'is_active' => 'boolean',
        ]);

        $store = Store::create($validated);

        return response()->json([
            'data' => $store,
        ], 201);
    }

    public function update(Request $request, $id)
    {
        $store = Store::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'street' => 'sometimes|required|string|max:255',
            'city' => 'sometimes|required|string|max:100',
            'postal_code' => 'sometimes|required|string|max:20',
            'phone' => 'sometimes|required|string|max:20',
            'email' => 'sometimes|required|email|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'is_active' => 'boolean',
        ]);

        $store->update($validated);

        return response()->json([
            'data' => $store,
        ]);
    }

    public function destroy($id)
    {
        $store = Store::findOrFail($id);

        // Check if store has stock or orders
        $hasStock = $store->stockItems()->exists();
        $hasOrders = $store->orders()->exists();

        if ($hasStock || $hasOrders) {
            $reasons = [];
            if ($hasStock) $reasons[] = 'des stocks';
            if ($hasOrders) $reasons[] = 'des commandes';

            return response()->json([
                'error_type' => 'store_has_data',
                'message' => 'Ce magasin ne peut pas être supprimé car il est lié à ' . implode(' et ', $reasons) . '. Vous pouvez le désactiver pour qu\'il ne soit plus visible sur le site.',
                'can_deactivate' => true,
            ], 400);
        }

        $store->delete();

        return response()->json([
            'message' => 'Magasin supprimé avec succès'
        ]);
    }

    /**
     * Toggle store active status
     */
    public function toggleActive($id)
    {
        $store = Store::findOrFail($id);
        $store->is_active = !$store->is_active;
        $store->save();

        return response()->json([
            'message' => $store->is_active ? 'Magasin activé avec succès' : 'Magasin désactivé avec succès',
            'is_active' => $store->is_active,
            'data' => $store,
        ]);
    }
}
