<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Store;
use App\Models\Category;
use App\Models\StockItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockController extends Controller
{
    /**
     * Get stock summary by store
     */
    public function getStoreStock($storeId)
    {
        $store = Store::findOrFail($storeId);

        // Get stock by category
        $stockByCategory = Category::withCount(['products as total_stock' => function ($query) use ($storeId) {
            $query->join('product_variants', 'products.id', '=', 'product_variants.product_id')
                ->join('stock_items', 'product_variants.id', '=', 'stock_items.variant_id')
                ->where('stock_items.store_id', $storeId)
                ->select(DB::raw('SUM(stock_items.quantity)'));
        }])->get();

        // Get total stock for this store
        $totalStock = StockItem::where('store_id', $storeId)->sum('quantity');

        // Get low stock items (below alert threshold)
        $lowStockItems = StockItem::where('store_id', $storeId)
            ->whereColumn('quantity', '<=', 'alert_threshold')
            ->with(['variant.product', 'product'])
            ->get();

        // Get stock by product with variants
        $stockByProduct = StockItem::where('store_id', $storeId)
            ->with(['product.category', 'variant'])
            ->get()
            ->groupBy('product.category.name')
            ->map(function ($items) {
                return [
                    'products' => $items->groupBy('product.name')->map(function ($productItems) {
                        return [
                            'product_id' => $productItems->first()->product_id,
                            'product_name' => $productItems->first()->product->name,
                            'variants' => $productItems->map(function ($item) {
                                return [
                                    'stock_item_id' => $item->id,
                                    'variant_id' => $item->variant_id,
                                    'size' => $item->variant->size,
                                    'quantity' => $item->quantity,
                                    'reserved' => $item->reserved_quantity,
                                    'available' => $item->quantity - $item->reserved_quantity,
                                    'alert_threshold' => $item->alert_threshold,
                                    'is_low_stock' => $item->quantity <= $item->alert_threshold,
                                ];
                            })->values(),
                            'total_quantity' => $productItems->sum('quantity'),
                        ];
                    })->values(),
                    'total_quantity' => $items->sum('quantity'),
                ];
            });

        return response()->json([
            'store' => $store,
            'total_stock' => $totalStock,
            'stock_by_category' => $stockByCategory,
            'stock_by_product' => $stockByProduct,
            'low_stock_items' => $lowStockItems,
            'low_stock_count' => $lowStockItems->count(),
        ]);
    }

    /**
     * Get global stock statistics
     */
    public function getGlobalStats()
    {
        // Total stock across all stores
        $totalStock = StockItem::sum('quantity');

        // Stock by store
        $stockByStore = Store::with(['stockItems'])->get()->map(function ($store) {
            return [
                'store_id' => $store->id,
                'store_name' => $store->name,
                'city' => $store->city,
                'total_stock' => $store->stockItems->sum('quantity'),
                'low_stock_count' => $store->stockItems->filter(function ($item) {
                    return $item->quantity <= $item->alert_threshold;
                })->count(),
            ];
        });

        // Stock by category across all stores
        $stockByCategory = Category::withCount(['products as total_stock' => function ($query) {
            $query->join('product_variants', 'products.id', '=', 'product_variants.product_id')
                ->join('stock_items', 'product_variants.id', '=', 'stock_items.variant_id')
                ->select(DB::raw('SUM(stock_items.quantity)'));
        }])->get();

        // Global low stock items
        $lowStockCount = StockItem::whereColumn('quantity', '<=', 'alert_threshold')->count();

        return response()->json([
            'total_stock' => $totalStock,
            'stock_by_store' => $stockByStore,
            'stock_by_category' => $stockByCategory,
            'low_stock_count' => $lowStockCount,
        ]);
    }

    /**
     * Get stock alerts
     */
    public function getAlerts()
    {
        $alerts = StockItem::whereColumn('quantity', '<=', 'alert_threshold')
            ->with(['product.category', 'variant', 'store'])
            ->get()
            ->map(function ($item) {
                return [
                    'product_name' => $item->product->name,
                    'category' => $item->product->category->name,
                    'variant_size' => $item->variant->size,
                    'store_name' => $item->store->name,
                    'store_city' => $item->store->city,
                    'quantity' => $item->quantity,
                    'alert_threshold' => $item->alert_threshold,
                    'severity' => $item->quantity == 0 ? 'critical' : ($item->quantity < 3 ? 'high' : 'medium'),
                ];
            });

        return response()->json($alerts);
    }

    /**
     * Adjust stock manually
     */
    public function adjustStock(Request $request)
    {
        $validated = $request->validate([
            'stock_item_id' => 'required|exists:stock_items,id',
            'adjustment' => 'required|integer',
            'reason' => 'nullable|string|max:500',
        ]);

        $stockItem = StockItem::findOrFail($validated['stock_item_id']);

        // Calculate new quantity
        $newQuantity = $stockItem->quantity + $validated['adjustment'];

        if ($newQuantity < 0) {
            return response()->json([
                'message' => 'Adjustment would result in negative stock. Current quantity: ' . $stockItem->quantity
            ], 400);
        }

        // Update stock
        $oldQuantity = $stockItem->quantity;
        $stockItem->quantity = $newQuantity;
        $stockItem->save();

        // Log the adjustment (you can implement a stock_movements table later)
        \Log::info('Stock adjustment', [
            'stock_item_id' => $stockItem->id,
            'product_id' => $stockItem->product_id,
            'variant_id' => $stockItem->variant_id,
            'store_id' => $stockItem->store_id,
            'old_quantity' => $oldQuantity,
            'adjustment' => $validated['adjustment'],
            'new_quantity' => $newQuantity,
            'reason' => $validated['reason'] ?? 'Manual adjustment',
            'user' => auth()->user()->email ?? 'system',
        ]);

        return response()->json([
            'message' => 'Stock adjusted successfully',
            'stock_item' => $stockItem->load(['product', 'variant', 'store']),
            'old_quantity' => $oldQuantity,
            'new_quantity' => $newQuantity,
        ]);
    }
}
