<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;

class SettingsController extends Controller
{
    /**
     * Get all settings
     */
    public function index()
    {
        $settings = Setting::all();

        // Transform into key-value pairs with proper types
        $settingsArray = [];
        foreach ($settings as $setting) {
            $settingsArray[$setting->key] = [
                'value' => Setting::get($setting->key),
                'type' => $setting->type,
                'description' => $setting->description,
            ];
        }

        return response()->json($settingsArray);
    }

    /**
     * Get a specific setting by key
     */
    public function show($key)
    {
        $value = Setting::get($key);

        if ($value === null) {
            return response()->json([
                'message' => 'Setting not found',
            ], 404);
        }

        return response()->json([
            'key' => $key,
            'value' => $value,
        ]);
    }

    /**
     * Update multiple settings at once
     */
    public function updateBatch(Request $request)
    {
        $validated = $request->validate([
            'settings' => 'required|array',
            'settings.*.key' => 'required|string',
            'settings.*.value' => 'required',
            'settings.*.type' => 'required|in:string,integer,boolean,json',
            'settings.*.description' => 'nullable|string',
        ]);

        foreach ($validated['settings'] as $settingData) {
            Setting::set(
                $settingData['key'],
                $settingData['value'],
                $settingData['type'],
                $settingData['description'] ?? null
            );
        }

        return response()->json([
            'message' => 'Settings updated successfully',
        ]);
    }

    /**
     * Update a single setting
     */
    public function update(Request $request, $key)
    {
        $validated = $request->validate([
            'value' => 'required',
            'type' => 'required|in:string,integer,boolean,json',
            'description' => 'nullable|string',
        ]);

        Setting::set(
            $key,
            $validated['value'],
            $validated['type'],
            $validated['description'] ?? null
        );

        return response()->json([
            'message' => 'Setting updated successfully',
            'setting' => [
                'key' => $key,
                'value' => Setting::get($key),
            ],
        ]);
    }

    /**
     * Delete a setting
     */
    public function destroy($key)
    {
        $setting = Setting::where('key', $key)->first();

        if (!$setting) {
            return response()->json([
                'message' => 'Setting not found',
            ], 404);
        }

        $setting->delete();

        return response()->json([
            'message' => 'Setting deleted successfully',
        ]);
    }
}
