<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Promotion;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PromotionController extends Controller
{
    /**
     * List all promotions
     */
    public function index(Request $request)
    {
        $query = Promotion::query();

        // Filter by active status
        if ($request->has('active_only') && $request->active_only) {
            $query->active();
        }

        $promotions = $query->orderBy('created_at', 'desc')->get();

        // Add validation status to each promotion
        $promotions->transform(function ($promotion) {
            $promotion->is_valid = $promotion->isValid();
            return $promotion;
        });

        return response()->json($promotions);
    }

    /**
     * Get active promotions only (for client display)
     */
    public function active()
    {
        $promotions = Promotion::active()->get();
        return response()->json($promotions);
    }

    /**
     * Get a specific promotion
     */
    public function show($id)
    {
        $promotion = Promotion::findOrFail($id);
        $promotion->is_valid = $promotion->isValid();
        return response()->json($promotion);
    }

    /**
     * Create a new promotion
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
        ]);

        $promotion = Promotion::create($validated);

        return response()->json([
            'message' => 'Promotion created successfully',
            'promotion' => $promotion,
        ], 201);
    }

    /**
     * Update a promotion
     */
    public function update(Request $request, $id)
    {
        $promotion = Promotion::findOrFail($id);

        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|string',
            'start_date' => 'sometimes|date',
            'end_date' => 'sometimes|date|after:start_date',
            'is_active' => 'sometimes|boolean',
        ]);

        $promotion->update($validated);

        return response()->json([
            'message' => 'Promotion updated successfully',
            'promotion' => $promotion->fresh(),
        ]);
    }

    /**
     * Toggle promotion active status
     */
    public function toggleActive($id)
    {
        $promotion = Promotion::findOrFail($id);

        $promotion->is_active = !$promotion->is_active;
        $promotion->save();

        return response()->json([
            'message' => 'Promotion status updated successfully',
            'promotion' => $promotion,
        ]);
    }

    /**
     * Delete a promotion
     */
    public function destroy($id)
    {
        $promotion = Promotion::findOrFail($id);
        $promotion->delete();

        return response()->json([
            'message' => 'Promotion deleted successfully',
        ]);
    }
}
