<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\OrderService;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    protected $orderService;

    public function __construct(OrderService $orderService)
    {
        $this->orderService = $orderService;
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'items' => 'required|array',
            'items.*.variant_id' => 'required|exists:product_variants,id',
            'items.*.quantity' => 'required|integer|min:1',
            'store_id' => 'required|exists:stores,id',
            'customer_first_name' => 'required|string',
            'customer_last_name' => 'required|string',
            'customer_phone' => 'required|string',
            'customer_email' => 'required|email',
            'shipping_address' => 'required|string',
            'shipping_city' => 'nullable|string',
            'shipping_postal_code' => 'nullable|string',
        ]);

        try {
            // Mapper shipping_* → delivery_* pour compatibilité avec OrderService
            $orderData = [
                'items' => $validated['items'],
                'store_id' => $validated['store_id'],
                'customer_first_name' => $validated['customer_first_name'],
                'customer_last_name' => $validated['customer_last_name'],
                'customer_phone' => $validated['customer_phone'],
                'customer_email' => $validated['customer_email'],
                'delivery_street' => $validated['shipping_address'],
                'delivery_city' => $validated['shipping_city'] ?? null,
                'delivery_postal_code' => $validated['shipping_postal_code'] ?? null,
                'user_id' => auth()->id() ?? null, // Si utilisateur authentifié
            ];

            $order = $this->orderService->createOrder($orderData);

            return response()->json([
                'data' => $order
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to create order',
                'error' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Afficher une commande via UUID (accès public sécurisé)
     *
     * @param string $uuid UUID de la commande
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($uuid)
    {
        $order = Order::where('uuid', $uuid)
            ->with(['items.variant.product.category', 'store'])
            ->firstOrFail();

        // Si utilisateur authentifié et commande a un user_id, vérifier propriété
        if (auth()->check() && $order->user_id !== null) {
            if ($order->user_id !== auth()->id()) {
                abort(403, 'Accès interdit à cette commande');
            }
        }

        return response()->json($order);
    }

    /**
     * List all orders (admin only)
     */
    public function index(Request $request)
    {
        $query = Order::with(['store', 'items']);

        // Filter by status
        if ($request->has('status') && $request->status !== 'ALL') {
            $query->where('status', $request->status);
        }

        // Search by order number or customer
        if ($request->has('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('order_number', 'like', '%' . $request->search . '%')
                  ->orWhere('customer_first_name', 'like', '%' . $request->search . '%')
                  ->orWhere('customer_last_name', 'like', '%' . $request->search . '%')
                  ->orWhere('customer_email', 'like', '%' . $request->search . '%');
            });
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(50);

        // Add items count to each order
        $orders->getCollection()->transform(function ($order) {
            $order->items_count = $order->items->count();
            return $order;
        });

        return response()->json($orders);
    }

    /**
     * Update order status (admin only)
     */
    public function updateStatus(Request $request, $id)
    {
        $validated = $request->validate([
            'status' => 'required|in:EN_ATTENTE_TRAITEMENT,EN_ATTENTE_LIVRAISON,LIVREE,ANNULEE',
            'comment' => 'nullable|string',
        ]);

        $order = Order::findOrFail($id);
        $newStatus = $validated['status'];
        $comment = $request->input('comment');

        try {
            // Utiliser la bonne méthode selon le statut
            if ($newStatus === 'LIVREE') {
                // Utiliser deliverOrder() qui gère le stock physique
                $order = $this->orderService->deliverOrder(
                    $order->id,
                    $comment,
                    auth()->id()
                );
            } elseif ($newStatus === 'ANNULEE') {
                // Utiliser cancelOrder() qui libère la réservation
                $order = $this->orderService->cancelOrder(
                    $order->id,
                    $comment,
                    auth()->id()
                );
            } else {
                // Pour EN_ATTENTE_LIVRAISON : juste changer le statut
                $order->updateStatus($newStatus, $comment, auth()->id());
                $order = $order->fresh(['store', 'items']);
            }

            return response()->json([
                'message' => 'Order status updated successfully',
                'order' => $order,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to update order status',
                'error' => $e->getMessage()
            ], 422);
        }
    }
}
