<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Store;
use App\Models\Order;
use App\Models\StockItem;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics
     */
    public function stats()
    {
        // Products statistics
        $productsStats = Product::selectRaw('
            COUNT(*) as total,
            SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive
        ')->first();

        // Stores statistics
        $storesStats = Store::selectRaw('
            COUNT(*) as total,
            SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive
        ')->first();

        // Orders statistics
        $ordersStats = Order::selectRaw('
            COUNT(*) as total,
            SUM(CASE WHEN status = "LIVREE" THEN total_amount ELSE 0 END) as total_revenue,
            SUM(CASE WHEN status = "EN_ATTENTE_TRAITEMENT" THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = "EN_ATTENTE_LIVRAISON" THEN 1 ELSE 0 END) as shipping,
            SUM(CASE WHEN status = "LIVREE" THEN 1 ELSE 0 END) as delivered,
            SUM(CASE WHEN status = "ANNULEE" THEN 1 ELSE 0 END) as cancelled
        ')->first();

        // Stock statistics
        $stockStats = StockItem::selectRaw('
            COUNT(*) as total_items,
            SUM(quantity) as total_quantity,
            SUM(reserved_quantity) as total_reserved,
            SUM(CASE WHEN quantity <= alert_threshold THEN 1 ELSE 0 END) as low_stock_count
        ')->first();

        // Stock by store
        $stockByStore = StockItem::join('stores', 'stock_items.store_id', '=', 'stores.id')
            ->where('stores.is_active', true)
            ->selectRaw('
                stores.id,
                stores.name,
                COUNT(stock_items.id) as items_count,
                SUM(stock_items.quantity) as total_quantity
            ')
            ->groupBy('stores.id', 'stores.name')
            ->get();

        // Stock by product
        $stockByProduct = StockItem::join('products', 'stock_items.product_id', '=', 'products.id')
            ->where('products.is_active', true)
            ->selectRaw('
                products.id,
                products.name,
                SUM(stock_items.quantity) as total_quantity,
                COUNT(DISTINCT stock_items.store_id) as stores_count
            ')
            ->groupBy('products.id', 'products.name')
            ->orderBy('total_quantity', 'desc')
            ->limit(10)
            ->get();

        // Low stock alerts
        $lowStockAlerts = StockItem::where('quantity', '<=', DB::raw('alert_threshold'))
            ->with(['product:id,name', 'variant:id,sku', 'store:id,name'])
            ->orderBy('quantity', 'asc')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'product_name' => $item->product->name,
                    'variant_sku' => $item->variant->sku,
                    'store_name' => $item->store->name,
                    'quantity' => $item->quantity,
                    'alert_threshold' => $item->alert_threshold,
                ];
            });

        // Recent orders (last 5)
        $recentOrders = Order::with(['user:id,first_name,last_name,email'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'customer_name' => $order->customer_first_name . ' ' . $order->customer_last_name,
                    'customer_email' => $order->customer_email,
                    'total_amount' => $order->total_amount,
                    'status' => $order->status,
                    'created_at' => $order->created_at->format('Y-m-d H:i:s'),
                ];
            });

        return response()->json([
            'data' => [
                'products' => [
                    'total' => $productsStats->total ?? 0,
                    'active' => $productsStats->active ?? 0,
                    'inactive' => $productsStats->inactive ?? 0,
                ],
                'stores' => [
                    'total' => $storesStats->total ?? 0,
                    'active' => $storesStats->active ?? 0,
                    'inactive' => $storesStats->inactive ?? 0,
                ],
                'orders' => [
                    'total' => $ordersStats->total ?? 0,
                    'total_revenue' => floatval($ordersStats->total_revenue ?? 0),
                    'pending' => $ordersStats->pending ?? 0,
                    'shipping' => $ordersStats->shipping ?? 0,
                    'delivered' => $ordersStats->delivered ?? 0,
                    'cancelled' => $ordersStats->cancelled ?? 0,
                ],
                'stock' => [
                    'total_items' => $stockStats->total_items ?? 0,
                    'total_quantity' => $stockStats->total_quantity ?? 0,
                    'total_reserved' => $stockStats->total_reserved ?? 0,
                    'low_stock_count' => $stockStats->low_stock_count ?? 0,
                    'by_store' => $stockByStore,
                    'by_product' => $stockByProduct,
                    'low_stock_alerts' => $lowStockAlerts,
                ],
                'recent_orders' => $recentOrders,
            ],
        ]);
    }
}
