<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StockInventory;
use App\Models\StockInventoryLine;
use App\Models\StockItem;
use App\Models\Store;
use App\Models\ProductVariant;
use App\Services\StockService;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\Rule;

class StockInventoryController extends Controller
{
    public function __construct(
        private StockService $stockService
    ) {}

    public function index(Request $request): View
    {
        $query = StockInventory::with('store');

        if ($request->filled('store_id')) {
            $query->where('store_id', $request->store_id);
        }

        if ($request->filled('month')) {
            $query->where('month', $request->month);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $inventories = $query->orderBy('month', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(20)
            ->withQueryString();

        $stores = Store::active()->get();

        return view('admin.stock.inventories.index', [
            'inventories' => $inventories,
            'stores' => $stores,
            'currentFilters' => [
                'store_id' => $request->store_id,
                'month' => $request->month,
                'status' => $request->status,
            ],
        ]);
    }

    public function create(): View
    {
        $stores = Store::active()->get();
        $defaultMonth = now()->format('Y-m');

        return view('admin.stock.inventories.create', [
            'stores' => $stores,
            'defaultMonth' => $defaultMonth,
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'store_id' => ['required', 'exists:stores,id'],
            'month' => [
                'required',
                'date_format:Y-m',
                Rule::unique('stock_inventories', 'month')->where(function ($query) use ($request) {
                    return $query->where('store_id', $request->store_id);
                }),
            ],
        ], [
            'month.unique' => 'Un inventaire existe deja pour ce magasin et ce mois.',
        ]);

        $inventory = StockInventory::create([
            'store_id' => $validated['store_id'],
            'month' => $validated['month'],
            'status' => StockInventory::STATUS_IN_PROGRESS,
        ]);

        $variants = ProductVariant::with('product')
            ->active()
            ->whereHas('product', function ($query) {
                $query->where('is_active', true);
            })
            ->orderBy('product_id')
            ->orderBy('id')
            ->get();

        foreach ($variants as $variant) {
            $stockItem = StockItem::firstOrCreate(
                [
                    'store_id' => $validated['store_id'],
                    'product_id' => $variant->product_id,
                    'variant_id' => $variant->id,
                ],
                [
                    'quantity' => 0,
                    'reserved_quantity' => 0,
                    'alert_threshold' => 5,
                ]
            );

            StockInventoryLine::create([
                'stock_inventory_id' => $inventory->id,
                'product_id' => $stockItem->product_id,
                'variant_id' => $stockItem->variant_id,
                'theoretical_qty' => $stockItem->quantity,
                'real_qty' => null,
                'gap' => null,
            ]);
        }

        return redirect()->route('admin.stock.inventories.show', $inventory->id)
            ->with('success', 'Inventaire cree avec succes');
    }

    public function show(int $id): View
    {
        $inventory = StockInventory::with(['store', 'lines.product', 'lines.variant'])
            ->findOrFail($id);

        $lines = $inventory->lines->sortBy([
            ['product_id', 'asc'],
            ['variant_id', 'asc'],
        ]);

        return view('admin.stock.inventories.show', [
            'inventory' => $inventory,
            'lines' => $lines,
        ]);
    }

    public function saveLines(Request $request, int $id): RedirectResponse
    {
        $inventory = StockInventory::with('lines')->findOrFail($id);

        if ($inventory->status !== StockInventory::STATUS_IN_PROGRESS) {
            return redirect()->back()
                ->with('error', 'Inventaire deja valide.');
        }

        $validated = $request->validate([
            'real_qty' => 'array',
            'real_qty.*' => 'nullable|integer|min:0',
        ]);

        $realQtyInputs = $validated['real_qty'] ?? [];

        foreach ($inventory->lines as $line) {
            if (!array_key_exists((string) $line->id, $realQtyInputs)) {
                continue;
            }

            $inputValue = $realQtyInputs[$line->id];
            $realQty = ($inputValue === '' || $inputValue === null) ? null : (int) $inputValue;
            $gap = $realQty === null ? null : ($realQty - $line->theoretical_qty);

            $line->real_qty = $realQty;
            $line->gap = $gap;
            $line->save();
        }

        return redirect()->back()
            ->with('success', 'Inventaire mis a jour');
    }

    public function validateInventory(Request $request, int $id): RedirectResponse
    {
        $inventory = StockInventory::with(['lines', 'store'])->findOrFail($id);

        if ($inventory->status !== StockInventory::STATUS_IN_PROGRESS) {
            return redirect()->back()
                ->with('error', 'Inventaire deja valide.');
        }

        $validated = $request->validate([
            'real_qty' => 'required|array',
            'real_qty.*' => 'required|integer|min:0',
        ]);

        $realQtyInputs = $validated['real_qty'] ?? [];

        if (count($realQtyInputs) !== $inventory->lines->count()) {
            return redirect()->back()
                ->with('error', 'Veuillez renseigner toutes les quantites reelles avant validation.');
        }

        $monthLabel = $this->formatMonthLabel($inventory->month);
        $reason = "Inventaire {$monthLabel} - Magasin {$inventory->store->name}";

        foreach ($inventory->lines as $line) {
            if (!array_key_exists((string) $line->id, $realQtyInputs)) {
                return redirect()->back()
                    ->with('error', 'Veuillez renseigner toutes les quantites reelles avant validation.');
            }

            $realQty = (int) $realQtyInputs[$line->id];
            $gap = $realQty - $line->theoretical_qty;

            $line->real_qty = $realQty;
            $line->gap = $gap;
            $line->save();

            if ($gap !== 0) {
                $stockItem = StockItem::where('store_id', $inventory->store_id)
                    ->where('product_id', $line->product_id)
                    ->where('variant_id', $line->variant_id)
                    ->first();

                if (!$stockItem) {
                    $stockItem = StockItem::create([
                        'store_id' => $inventory->store_id,
                        'product_id' => $line->product_id,
                        'variant_id' => $line->variant_id,
                        'quantity' => 0,
                        'reserved_quantity' => 0,
                        'alert_threshold' => 5,
                    ]);
                }

                $this->stockService->adjustStock(
                    $stockItem->id,
                    $realQty,
                    $reason,
                    auth()->id()
                );
            }
        }

        $inventory->status = StockInventory::STATUS_VALIDATED;
        $inventory->validated_at = now();
        $inventory->save();

        return redirect()->route('admin.stock.inventories.show', $inventory->id)
            ->with('success', 'Inventaire valide et ajustements appliques');
    }

    public function exportCsv(int $id)
    {
        $inventory = StockInventory::with(['store', 'lines.product', 'lines.variant'])
            ->findOrFail($id);

        $fileName = 'inventaire_' . $inventory->store_id . '_' . $inventory->month . '.csv';

        return Response::streamDownload(function () use ($inventory) {
            $handle = fopen('php://output', 'w');

            fputcsv($handle, [
                'Magasin',
                'Mois',
                'Produit',
                'Variante',
                'Theorique',
                'Reel',
                'Ecart',
                'Date validation',
            ]);

            foreach ($inventory->lines as $line) {
                fputcsv($handle, [
                    $inventory->store->name ?? '',
                    $inventory->month,
                    $line->product->name ?? '',
                    $line->variant->size ?? ($line->variant->name ?? ''),
                    $line->theoretical_qty,
                    $line->real_qty,
                    $line->gap,
                    $inventory->validated_at ? $inventory->validated_at->format('Y-m-d H:i') : '',
                ]);
            }

            fclose($handle);
        }, $fileName, [
            'Content-Type' => 'text/csv',
        ]);
    }

    private function formatMonthLabel(string $month): string
    {
        if (strlen($month) !== 7 || strpos($month, '-') === false) {
            return $month;
        }

        [$year, $monthPart] = explode('-', $month);
        return $monthPart . '/' . $year;
    }
}
