<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StockItem;
use App\Models\StockMovement;
use App\Models\Store;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Category;
use App\Services\StockService;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;

class StockController extends Controller
{
    public function __construct(
        private StockService $stockService
    ) {}

    public function index(Request $request): View
    {
        $query = StockItem::with(['product', 'variant', 'store']);

        if ($request->filled('store_id')) {
            $query->where('store_id', $request->store_id);
        }

        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        if ($request->filled('category_id')) {
            $query->whereHas('product', function ($subQuery) use ($request) {
                $subQuery->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('low_stock')) {
            $query->whereColumn('quantity', '<=', 'alert_threshold');
        }

        $stockItems = $query->orderBy('quantity', 'asc')->paginate(30)->withQueryString();
        $stores = Store::active()->get();
        $products = Product::active()->get();
        $categories = Category::orderBy('name')->get();

        // Global stats
        $stats = [
            'total_items' => StockItem::count(),
            'total_quantity' => StockItem::sum('quantity'),
            'total_reserved' => StockItem::sum('reserved_quantity'),
            'low_stock_count' => StockItem::whereColumn('quantity', '<=', 'alert_threshold')->count(),
        ];

        $globalQuery = StockItem::query()
            ->select(
                'product_id',
                'variant_id',
                DB::raw('SUM(quantity) as total_quantity'),
                DB::raw('SUM(reserved_quantity) as total_reserved'),
                DB::raw('SUM(quantity - reserved_quantity) as total_available'),
                DB::raw('MIN(alert_threshold) as min_alert_threshold')
            );

        if ($request->filled('product_id')) {
            $globalQuery->where('product_id', $request->product_id);
        }

        if ($request->filled('category_id')) {
            $globalQuery->whereIn('product_id', function ($subQuery) use ($request) {
                $subQuery->select('id')
                    ->from('products')
                    ->where('category_id', $request->category_id);
            });
        }

        $globalStock = $globalQuery
            ->groupBy('product_id', 'variant_id')
            ->orderBy('total_quantity', 'asc')
            ->get();

        $globalProductIds = $globalStock->pluck('product_id')->unique()->all();
        $globalVariantIds = $globalStock->pluck('variant_id')->unique()->all();
        $globalProducts = Product::whereIn('id', $globalProductIds)->get()->keyBy('id');
        $globalVariants = ProductVariant::whereIn('id', $globalVariantIds)->get()->keyBy('id');

        return view('admin.stock.index', [
            'stockItems' => $stockItems,
            'stores' => $stores,
            'products' => $products,
            'categories' => $categories,
            'stats' => $stats,
            'globalStock' => $globalStock,
            'globalProducts' => $globalProducts,
            'globalVariants' => $globalVariants,
            'currentFilters' => [
                'store_id' => $request->store_id,
                'product_id' => $request->product_id,
                'category_id' => $request->category_id,
                'low_stock' => $request->low_stock,
            ],
        ]);
    }

    public function adjust(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'stock_item_id' => 'required|exists:stock_items,id',
            'adjustment' => 'required|integer',
            'reason' => 'nullable|string|max:500',
        ]);

        try {
            $stockItem = StockItem::findOrFail($validated['stock_item_id']);
            $newQuantity = $stockItem->quantity + $validated['adjustment'];

            if ($newQuantity < 0) {
                return redirect()->back()
                    ->with('error', 'Ajustement invalide : le stock ne peut pas devenir negatif');
            }

            $this->stockService->adjustStock(
                $stockItem->id,
                $newQuantity,
                $validated['reason'] ?? 'Ajustement manuel',
                auth()->id()
            );

            return redirect()->back()
                ->with('success', 'Stock ajuste avec succes');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erreur: ' . $e->getMessage());
        }
    }

    public function create(): View
    {
        $stores = Store::active()->get();
        $products = Product::with('variants')->active()->get();

        return view('admin.stock.create', [
            'stores' => $stores,
            'products' => $products,
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'variant_id' => 'required|exists:product_variants,id',
            'store_id' => 'required|exists:stores,id',
            'quantity' => 'required|integer|min:0',
            'alert_threshold' => 'nullable|integer|min:0',
        ]);

        // Check if stock item already exists
        $existing = StockItem::where('product_id', $validated['product_id'])
            ->where('variant_id', $validated['variant_id'])
            ->where('store_id', $validated['store_id'])
            ->first();

        if ($existing) {
            return redirect()->back()
                ->with('error', 'Un stock existe deja pour cette combinaison produit/variante/magasin');
        }

        StockItem::create([
            'product_id' => $validated['product_id'],
            'variant_id' => $validated['variant_id'],
            'store_id' => $validated['store_id'],
            'quantity' => $validated['quantity'],
            'reserved_quantity' => 0,
            'alert_threshold' => $validated['alert_threshold'] ?? 5,
        ]);

        return redirect()->route('admin.stock.index')
            ->with('success', 'Stock cree avec succes');
    }

    public function movements(Request $request): View
    {
        $movementQuery = StockMovement::with([
            'stockItem.product',
            'stockItem.variant',
            'stockItem.store',
            'performedBy',
        ]);

        if ($request->filled('store_id')) {
            $movementQuery->whereHas('stockItem', function ($subQuery) use ($request) {
                $subQuery->where('store_id', $request->store_id);
            });
        }

        if ($request->filled('product_id')) {
            $movementQuery->whereHas('stockItem', function ($subQuery) use ($request) {
                $subQuery->where('product_id', $request->product_id);
            });
        }

        if ($request->filled('type')) {
            $movementQuery->where('type', $request->type);
        }

        if ($request->filled('date_from')) {
            $movementQuery->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $movementQuery->whereDate('created_at', '<=', $request->date_to);
        }

        $movements = $movementQuery
            ->orderBy('created_at', 'desc')
            ->paginate(30)
            ->withQueryString();

        $stores = Store::active()->get();
        $products = Product::active()->get();
        $variants = ProductVariant::with('product')->orderBy('id')->get();

        return view('admin.stock.movements', [
            'movements' => $movements,
            'stores' => $stores,
            'products' => $products,
            'variants' => $variants,
            'currentFilters' => [
                'store_id' => $request->store_id,
                'product_id' => $request->product_id,
                'type' => $request->type,
                'date_from' => $request->date_from,
                'date_to' => $request->date_to,
            ],
        ]);
    }

    public function storeMovement(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'store_id' => 'required|exists:stores,id',
            'product_id' => 'required|exists:products,id',
            'variant_id' => 'required|exists:product_variants,id',
            'type' => 'required|in:IN,OUT,ADJUSTMENT',
            'quantity' => 'required|integer',
            'reason' => 'nullable|string|max:500',
        ]);

        $quantity = (int) $validated['quantity'];

        if ($validated['type'] !== 'ADJUSTMENT' && $quantity <= 0) {
            return redirect()->back()
                ->withErrors(['quantity' => 'La quantite doit etre superieure a 0.'])
                ->withInput();
        }

        if ($validated['type'] === 'ADJUSTMENT' && $quantity === 0) {
            return redirect()->back()
                ->withErrors(['quantity' => "La quantite d'ajustement ne peut pas etre nulle."])
                ->withInput();
        }

        $variant = ProductVariant::where('id', $validated['variant_id'])
            ->where('product_id', $validated['product_id'])
            ->first();

        if (!$variant) {
            return redirect()->back()
                ->withErrors(['variant_id' => 'La variante selectionnee ne correspond pas au produit.'])
                ->withInput();
        }

        $defaultReason = match ($validated['type']) {
            'IN' => 'Entree manuelle',
            'OUT' => 'Sortie manuelle',
            'ADJUSTMENT' => 'Ajustement manuel',
            default => null,
        };

        $reason = $validated['reason'] ?? $defaultReason;

        try {
            if ($validated['type'] === 'IN') {
                $this->stockService->incrementStock(
                    $validated['product_id'],
                    $validated['variant_id'],
                    $validated['store_id'],
                    $quantity,
                    $reason,
                    auth()->id()
                );
            } elseif ($validated['type'] === 'OUT') {
                $this->stockService->decrementStock(
                    $validated['product_id'],
                    $validated['variant_id'],
                    $validated['store_id'],
                    $quantity,
                    auth()->id(),
                    $reason
                );
            } else {
                $stockItem = StockItem::where('product_id', $validated['product_id'])
                    ->where('variant_id', $validated['variant_id'])
                    ->where('store_id', $validated['store_id'])
                    ->first();

                if (!$stockItem) {
                    return redirect()->back()
                        ->withErrors(['store_id' => 'Stock introuvable pour ce magasin/produit/variante.'])
                        ->withInput();
                }

                $newQuantity = $stockItem->quantity + $quantity;

                if ($newQuantity < 0) {
                    return redirect()->back()
                        ->withErrors(['quantity' => 'Ajustement invalide: le stock ne peut pas devenir negatif.'])
                        ->withInput();
                }

                $this->stockService->adjustStock(
                    $stockItem->id,
                    $newQuantity,
                    $reason,
                    auth()->id()
                );
            }

            return redirect()->route('admin.stock.movements.index')
                ->with('success', 'Mouvement cree avec succes');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erreur: ' . $e->getMessage())
                ->withInput();
        }
    }
}
