<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Services\MailSettingsService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Support\Facades\Mail;

class SettingController extends Controller
{
    /**
     * Cles de parametres gerables depuis l'admin.
     * Format : key => [type, description, default]
     */
    private const MANAGED_SETTINGS = [
        'site_name' => ['string', 'Nom du site', 'Boutique Matelas'],
        'contact_email' => ['string', 'Email de contact', 'contact@jolimatelas.com'],
        'contact_phone' => ['string', 'Telephone de contact', ''],
        'company_address' => ['string', "Adresse de l'entreprise", ''],
        'currency_symbol' => ['string', 'Symbole de devise', 'XAF'],
        'order_notification_enabled' => ['boolean', 'Activer les notifications de commande par email', true],
        'admin_notification_emails' => ['json', 'Emails admin pour notifications (JSON array)', '[]'],
        'low_stock_threshold' => ['integer', "Seuil d'alerte stock bas (par defaut)", 5],
        'mail_enabled' => ['boolean', 'Activer les emails sortants', false],
        'mail_mailer' => ['string', 'Mailer sortant', 'smtp'],
        'mail_host' => ['string', 'SMTP host', ''],
        'mail_port' => ['integer', 'SMTP port', 587],
        'mail_username' => ['string', 'SMTP username', ''],
        'mail_password' => ['string', 'SMTP password', ''],
        'mail_encryption' => ['string', 'SMTP encryption', 'tls'],
        'mail_from_address' => ['string', 'Adresse expediteur', ''],
        'mail_from_name' => ['string', 'Nom expediteur', ''],
        'mail_reply_to_address' => ['string', 'Reply-to address', ''],
        'mail_reply_to_name' => ['string', 'Reply-to name', ''],
    ];

    public function index(): View
    {
        $settings = [];

        foreach (self::MANAGED_SETTINGS as $key => [$type, $description, $default]) {
            $value = Setting::get($key, $default);
            if ($key === 'mail_password') {
                $value = '';
            }

            $settings[$key] = [
                'value' => $value,
                'type' => $type,
                'description' => $description,
            ];
        }

        return view('admin.settings.index', [
            'settings' => $settings,
        ]);
    }

    public function update(Request $request): RedirectResponse
    {
        $rules = [
            'mail_enabled' => 'nullable|boolean',
            'mail_mailer' => 'nullable|in:smtp',
            'mail_host' => 'required_if:mail_enabled,1|nullable|string',
            'mail_port' => 'required_if:mail_enabled,1|nullable|integer',
            'mail_username' => 'nullable|string',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|in:tls,ssl,none',
            'mail_from_address' => 'required_if:mail_enabled,1|nullable|email',
            'mail_from_name' => 'nullable|string',
            'mail_reply_to_address' => 'nullable|email',
            'mail_reply_to_name' => 'nullable|string',
        ];

        $request->validate($rules, [
            'mail_host.required_if' => "Le host SMTP est obligatoire lorsque l'envoi d'email est active.",
            'mail_port.required_if' => "Le port SMTP est obligatoire lorsque l'envoi d'email est active.",
            'mail_from_address.required_if' => "L'adresse expediteur est obligatoire lorsque l'envoi d'email est active.",
        ]);

        foreach (self::MANAGED_SETTINGS as $key => [$type, $description, $default]) {
            if ($key === 'mail_password' && !$request->filled('mail_password')) {
                continue;
            }

            $value = $request->input($key);

            if ($key === 'mail_mailer' && !$value) {
                $value = 'smtp';
            }

            if ($key === 'mail_encryption' && ($value === 'none' || $value === null)) {
                $value = '';
            }

            if ($type === 'boolean') {
                $value = $request->boolean($key);
            }

            if ($type === 'json' && is_string($value)) {
                $decoded = json_decode($value, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', "Format JSON invalide pour \"{$description}\"");
                }
                $value = $decoded;
            }

            if ($type === 'integer') {
                $value = (int) $value;
            }

            Setting::set($key, $value, $type, $description);
        }

        return redirect()->route('admin.settings.index')
            ->with('success', 'Parametres enregistres avec succes');
    }

    public function sendTestEmail(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'test_email_to' => 'required|email',
        ]);

        $mailEnabled = (bool) Setting::get('mail_enabled', false);

        if (!$mailEnabled) {
            return redirect()->back()
                ->with('error', "Les emails sortants sont desactives.");
        }

        MailSettingsService::apply();

        try {
            Mail::raw('Test SMTP OK', function ($message) use ($validated) {
                $message->to($validated['test_email_to'])
                    ->subject('Test SMTP OK');
            });
        } catch (\Throwable $e) {
            return redirect()->back()
                ->with('error', "Echec d'envoi: " . $e->getMessage());
        }

        return redirect()->back()
            ->with('success', 'Email de test envoye avec succes');
    }
}