<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Promotion;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;

class PromotionController extends Controller
{
    public function index(Request $request): View
    {
        $query = Promotion::query();

        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        if ($request->filled('active')) {
            $query->where('is_active', $request->active === 'true');
        }

        $promotions = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();

        return view('admin.promotions.index', [
            'promotions' => $promotions,
            'currentFilters' => [
                'search' => $request->search,
                'active' => $request->active,
            ],
        ]);
    }

    public function create(): View
    {
        return view('admin.promotions.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'required|image|max:4096',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['start_date'] = now()->toDateString();
        $validated['end_date'] = now()->addYears(10)->toDateString();

        if ($request->hasFile('image')) {
            $validated['image_path'] = $request->file('image')->store('promotions', 'public');
        }

        Promotion::create($validated);

        return redirect()->route('admin.promotions.index')
            ->with('success', 'Promotion creee avec succes');
    }

    public function edit(int $id): View
    {
        $promotion = Promotion::findOrFail($id);

        return view('admin.promotions.edit', [
            'promotion' => $promotion,
        ]);
    }

    public function update(Request $request, int $id): RedirectResponse
    {
        $promotion = Promotion::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|max:4096',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        if ($request->hasFile('image')) {
            if ($promotion->image_path) {
                Storage::disk('public')->delete($promotion->image_path);
            }
            $validated['image_path'] = $request->file('image')->store('promotions', 'public');
        }

        $promotion->update($validated);

        return redirect()->route('admin.promotions.edit', $id)
            ->with('success', 'Promotion mise a jour avec succes');
    }

    public function destroy(int $id): RedirectResponse
    {
        $promotion = Promotion::findOrFail($id);

        if ($promotion->image_path) {
            Storage::disk('public')->delete($promotion->image_path);
        }

        $promotion->delete();

        return redirect()->route('admin.promotions.index')
            ->with('success', 'Promotion supprimee avec succes');
    }

    public function toggleActive(int $id): RedirectResponse
    {
        $promotion = Promotion::findOrFail($id);
        $promotion->is_active = !$promotion->is_active;
        $promotion->save();

        return redirect()->back()
            ->with('success', 'Statut de la promotion mis a jour');
    }
}
