<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\ProductVariant;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class ProductController extends Controller
{
    public function index(Request $request): View
    {
        $query = Product::with(['category', 'variants']);

        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        if ($request->filled('active')) {
            $query->where('is_active', $request->active === 'true');
        }

        $products = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        $categories = Category::all();

        return view('admin.products.index', [
            'products' => $products,
            'categories' => $categories,
            'currentFilters' => [
                'search' => $request->search,
                'category' => $request->category,
                'active' => $request->active,
            ],
        ]);
    }

    public function create(): View
    {
        $categories = Category::with('attributes')->orderBy('name')->get();
        return view('admin.products.create', [
            'categories' => $categories,
            'attributeValues' => [],
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_desc' => 'nullable|string|max:500',
            'base_price' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'is_active' => 'sometimes|boolean',
            'is_featured' => 'sometimes|boolean',
            'thumbnail' => 'nullable|image|max:2048',
            'images' => 'nullable|array|max:5',
            'images.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['is_featured'] = $request->boolean('is_featured', false);

        $category = Category::with('attributes')->findOrFail($validated['category_id']);
        $attributeValues = $this->validateAttributeValues($category, $request->input('attribute_values', []));

        if ($request->hasFile('thumbnail')) {
            $validated['thumbnail'] = $request->file('thumbnail')->store('products', 'public');
        }

        if ($request->hasFile('images')) {
            $imagePaths = [];
            foreach ($request->file('images') as $image) {
                $imagePaths[] = $image->store('products', 'public');
            }
            $validated['images'] = $imagePaths;
        }

        $product = Product::create($validated);
        $this->syncAttributeValues($product, $category, $attributeValues);

        // Un produit actif sans variante ne peut pas être commandé
        if ($product->is_active && $product->variants()->count() === 0) {
            $product->update(['is_active' => false]);
            return redirect()->route('admin.products.edit', $product->id)
                ->with('warning', 'Produit créé mais désactivé. Vous devez créer au moins une variante avant de l\'activer.');
        }

        return redirect()->route('admin.products.index')
            ->with('success', 'Produit créé avec succès');
    }

    public function edit(int $id): View
    {
        $product = Product::with(['variants', 'attributeValues'])->findOrFail($id);
        $categories = Category::with('attributes')->orderBy('name')->get();
        $attributeValues = $product->attributeValues->pluck('value', 'attribute_id')->toArray();

        return view('admin.products.edit', [
            'product' => $product,
            'categories' => $categories,
            'attributeValues' => $attributeValues,
        ]);
    }

    public function update(Request $request, int $id): RedirectResponse
    {
        $product = Product::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_desc' => 'nullable|string|max:500',
            'base_price' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'is_active' => 'sometimes|boolean',
            'is_featured' => 'sometimes|boolean',
            'thumbnail' => 'nullable|image|max:2048',
            'images' => 'nullable|array|max:5',
            'images.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['is_featured'] = $request->boolean('is_featured', false);

        $category = Category::with('attributes')->findOrFail($validated['category_id']);
        $attributeValues = $this->validateAttributeValues($category, $request->input('attribute_values', []));

        if ($request->hasFile('thumbnail')) {
            if ($product->thumbnail) {
                Storage::disk('public')->delete($product->thumbnail);
            }
            $validated['thumbnail'] = $request->file('thumbnail')->store('products', 'public');
        }

        if ($request->hasFile('images')) {
            $imagePaths = [];
            foreach ($request->file('images') as $image) {
                $imagePaths[] = $image->store('products', 'public');
            }
            $validated['images'] = array_values(array_filter(array_merge($product->images ?? [], $imagePaths)));
        }

        // Bloquer activation si aucune variante
        if ($validated['is_active'] && $product->variants()->count() === 0) {
            $validated['is_active'] = false;
            $product->update($validated);
            $this->syncAttributeValues($product, $category, $attributeValues);

            return redirect()->route('admin.products.edit', $product->id)
                ->with('warning', 'Modifications enregistrées, mais le produit reste désactivé. Vous devez créer au moins une variante avant de l\'activer.');
        }

        $product->update($validated);
        $this->syncAttributeValues($product, $category, $attributeValues);

        return redirect()->route('admin.products.edit', $product->id)
            ->with('success', 'Produit mis à jour avec succès');
    }

    public function destroy(int $id): RedirectResponse
    {
        $product = Product::findOrFail($id);

        if ($product->thumbnail) {
            Storage::disk('public')->delete($product->thumbnail);
        }

        $product->delete();

        return redirect()->route('admin.products.index')
            ->with('success', 'Produit supprime avec succes');
    }

    public function toggleActive(int $id): RedirectResponse
    {
        $product = Product::findOrFail($id);

        // Bloquer activation si aucune variante
        if (!$product->is_active && $product->variants()->count() === 0) {
            return redirect()->back()
                ->with('error', 'Impossible d\'activer ce produit. Vous devez créer au moins une variante.');
        }

        $product->is_active = !$product->is_active;
        $product->save();

        return redirect()->back()
            ->with('success', 'Statut du produit mis à jour');
    }

    public function storeVariant(Request $request, int $productId): RedirectResponse
    {
        $product = Product::findOrFail($productId);

        $validated = $request->validate([
            'size' => 'required|string|max:50',
            'price' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $validated['product_id'] = $product->id;
        $validated['is_active'] = $request->boolean('is_active', true);

        ProductVariant::create($validated);

        return redirect()->route('admin.products.edit', $productId)
            ->with('success', 'Variante ajoutee avec succes');
    }

    public function destroyVariant(int $productId, int $variantId): RedirectResponse
    {
        $product = Product::findOrFail($productId);
        $variant = ProductVariant::where('product_id', $productId)
            ->findOrFail($variantId);

        // Empêcher la suppression de la dernière variante d'un produit actif
        if ($product->is_active && $product->variants()->count() <= 1) {
            return redirect()->route('admin.products.edit', $productId)
                ->with('error', 'Impossible de supprimer la dernière variante d\'un produit actif. Désactivez le produit d\'abord.');
        }

        $variant->delete();

        return redirect()->route('admin.products.edit', $productId)
            ->with('success', 'Variante supprimée avec succès');
    }

    private function validateAttributeValues(Category $category, array $inputValues): array
    {
        if ($category->attributes->isEmpty()) {
            return [];
        }

        $errors = [];
        $values = [];

        foreach ($category->attributes as $attribute) {
            $raw = $inputValues[$attribute->id] ?? null;
            $value = is_string($raw) ? trim($raw) : $raw;
            $hasValue = !is_null($value) && $value !== '';

            if ($attribute->required && !$hasValue) {
                $errors["attribute_values.{$attribute->id}"] = "Le champ {$attribute->name} est obligatoire.";
                continue;
            }

            if (!$hasValue) {
                continue;
            }

            if ($attribute->type === 'number' && !is_numeric($value)) {
                $errors["attribute_values.{$attribute->id}"] = "Le champ {$attribute->name} doit etre un nombre.";
                continue;
            }

            if ($attribute->type === 'select') {
                $options = $attribute->options ?? [];
                if (!in_array($value, $options, true)) {
                    $errors["attribute_values.{$attribute->id}"] = "Valeur invalide pour {$attribute->name}.";
                    continue;
                }
            }

            $values[$attribute->id] = (string) $value;
        }

        if ($errors) {
            throw ValidationException::withMessages($errors);
        }

        return $values;
    }

    private function syncAttributeValues(Product $product, Category $category, array $values): void
    {
        $attributeIds = $category->attributes->pluck('id')->all();

        if (empty($attributeIds)) {
            return;
        }

        $product->attributeValues()
            ->whereNotIn('attribute_id', $attributeIds)
            ->delete();

        $providedIds = array_keys($values);

        foreach ($values as $attributeId => $value) {
            $product->attributeValues()->updateOrCreate(
                ['attribute_id' => $attributeId],
                ['value' => $value]
            );
        }

        $product->attributeValues()
            ->whereIn('attribute_id', $attributeIds)
            ->whereNotIn('attribute_id', $providedIds)
            ->delete();
    }
}
