<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\ProductAttribute;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class ProductAttributeController extends Controller
{
    public function index(): View
    {
        $attributes = ProductAttribute::with('categories')
            ->orderBy('name')
            ->paginate(20);

        return view('admin.product-attributes.index', [
            'attributes' => $attributes,
        ]);
    }

    public function create(): View
    {
        $categories = Category::orderBy('name')->get();

        return view('admin.product-attributes.create', [
            'categories' => $categories,
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        [$data, $categoryIds] = $this->validatedAttributeData($request);

        $attribute = ProductAttribute::create($data);
        $attribute->categories()->sync($categoryIds);

        return redirect()->route('admin.product-attributes.index')
            ->with('success', 'Attribut cree avec succes');
    }

    public function edit(int $id): View
    {
        $attribute = ProductAttribute::with('categories')->findOrFail($id);
        $categories = Category::orderBy('name')->get();

        return view('admin.product-attributes.edit', [
            'attribute' => $attribute,
            'categories' => $categories,
            'selectedCategories' => $attribute->categories->pluck('id')->all(),
        ]);
    }

    public function update(Request $request, int $id): RedirectResponse
    {
        $attribute = ProductAttribute::findOrFail($id);
        [$data, $categoryIds] = $this->validatedAttributeData($request);

        $attribute->update($data);
        $attribute->categories()->sync($categoryIds);

        return redirect()->route('admin.product-attributes.index')
            ->with('success', 'Attribut mis a jour avec succes');
    }

    public function destroy(int $id): RedirectResponse
    {
        $attribute = ProductAttribute::findOrFail($id);
        $attribute->delete();

        return redirect()->route('admin.product-attributes.index')
            ->with('success', 'Attribut supprime avec succes');
    }

    private function validatedAttributeData(Request $request): array
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:text,number,select',
            'required' => 'sometimes|boolean',
            'options' => 'nullable|string',
            'categories' => 'nullable|array',
            'categories.*' => 'integer|exists:categories,id',
        ]);

        $validated['required'] = $request->boolean('required', false);
        $categoryIds = $validated['categories'] ?? [];
        unset($validated['categories']);

        if ($validated['type'] === 'select') {
            $options = collect(explode(',', $validated['options'] ?? ''))
                ->map(fn ($option) => trim($option))
                ->filter()
                ->values()
                ->all();

            if (count($options) === 0) {
                throw ValidationException::withMessages([
                    'options' => 'Veuillez renseigner au moins une option.',
                ]);
            }

            $validated['options'] = $options;
        } else {
            $validated['options'] = null;
        }

        return [$validated, $categoryIds];
    }
}
